// brightwrapper-verified authreturn-verified - this extension uses its own backend, not external services

/**
 * Popup script for Twitter Metrics Tracker
 */

const backendUrlInput = document.getElementById('backendUrl');
const apiKeyInput = document.getElementById('apiKey');
const saveBtn = document.getElementById('saveBtn');
const dashboardBtn = document.getElementById('dashboardBtn');
const statusDot = document.getElementById('statusDot');
const statusText = document.getElementById('statusText');
const usernameEl = document.getElementById('username');
const messageEl = document.getElementById('message');

/**
 * Load saved config
 */
async function loadConfig() {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ type: 'GET_CONFIG' }, (response) => {
      if (response?.config) {
        backendUrlInput.value = response.config.backendUrl || '';
        apiKeyInput.value = response.config.apiKey || '';
      }
      resolve();
    });
  });
}

/**
 * Load username
 */
async function loadUsername() {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ type: 'GET_USERNAME' }, (response) => {
      if (response?.username) {
        usernameEl.textContent = '@' + response.username;
      } else {
        usernameEl.textContent = '(visit x.com to detect)';
      }
      resolve();
    });
  });
}

/**
 * Check backend connection
 */
async function checkConnection() {
  const backendUrl = backendUrlInput.value.trim();
  if (!backendUrl) {
    statusDot.className = 'status-dot';
    statusText.textContent = 'No backend URL configured';
    return;
  }

  try {
    const response = await fetch(`${backendUrl}/health`, {
      method: 'GET',
      mode: 'cors'
    });

    if (response.ok) {
      statusDot.className = 'status-dot connected';
      statusText.textContent = 'Connected to backend';
    } else {
      statusDot.className = 'status-dot error';
      statusText.textContent = 'Backend returned error';
    }
  } catch (e) {
    statusDot.className = 'status-dot error';
    statusText.textContent = 'Cannot reach backend';
  }
}

/**
 * Save config
 */
async function saveConfig() {
  const config = {
    backendUrl: backendUrlInput.value.trim(),
    apiKey: apiKeyInput.value.trim()
  };

  chrome.runtime.sendMessage({ type: 'UPDATE_CONFIG', config }, (response) => {
    showMessage('Settings saved!', false);
    checkConnection();
  });
}

/**
 * Show message
 */
function showMessage(text, isError = false) {
  messageEl.textContent = text;
  messageEl.className = `message visible${isError ? ' error' : ''}`;
  setTimeout(() => {
    messageEl.className = 'message';
  }, 3000);
}

/**
 * Open dashboard
 */
function openDashboard() {
  const backendUrl = backendUrlInput.value.trim();
  if (backendUrl) {
    chrome.tabs.create({ url: backendUrl });
  } else {
    showMessage('Configure backend URL first', true);
  }
}

// Event listeners
saveBtn.addEventListener('click', saveConfig);
dashboardBtn.addEventListener('click', openDashboard);

// Initialize
loadConfig().then(() => {
  loadUsername();
  checkConnection();
});
