/**
 * Content script for Twitter Metrics Tracker
 * Runs on x.com/twitter.com pages to:
 * 1. Detect the logged-in user
 * 2. Intercept API responses containing analytics data
 * 3. Extract metrics from the page
 */

(function() {
  'use strict';

  console.log('[TwitterMetrics] Content script loaded');

  /**
   * Detect and save the logged-in username
   */
  function detectUsername() {
    // Method 1: Look for profile link in navigation
    const profileLink = document.querySelector('a[data-testid="AppTabBar_Profile_Link"]');
    if (profileLink) {
      const href = profileLink.getAttribute('href');
      if (href && href.startsWith('/')) {
        const username = href.substring(1).split('/')[0];
        if (username && !username.includes('/')) {
          saveUsername(username);
          return username;
        }
      }
    }

    // Method 2: Look for username in account switcher
    const accountSwitcher = document.querySelector('[data-testid="SideNav_AccountSwitcher_Button"]');
    if (accountSwitcher) {
      const usernameSpan = accountSwitcher.querySelector('span[dir="ltr"]');
      if (usernameSpan && usernameSpan.textContent.startsWith('@')) {
        const username = usernameSpan.textContent.substring(1);
        saveUsername(username);
        return username;
      }
    }

    return null;
  }

  function saveUsername(username) {
    chrome.runtime.sendMessage({ type: 'SET_USERNAME', username }, (response) => {
      console.log('[TwitterMetrics] Username saved:', username);
    });
  }

  /**
   * Intercept fetch to capture API responses
   */
  const originalFetch = window.fetch;
  window.fetch = async function(...args) {
    const response = await originalFetch.apply(this, args);

    try {
      const url = typeof args[0] === 'string' ? args[0] : args[0].url;

      // Check if this is an analytics-related API call
      if (isAnalyticsUrl(url)) {
        // Clone response to read body without consuming it
        const clone = response.clone();
        const data = await clone.json();
        processApiResponse(url, data);
      }
    } catch (e) {
      // Ignore parsing errors
    }

    return response;
  };

  /**
   * Intercept XMLHttpRequest
   */
  const originalXHROpen = XMLHttpRequest.prototype.open;
  const originalXHRSend = XMLHttpRequest.prototype.send;

  XMLHttpRequest.prototype.open = function(method, url, ...rest) {
    this._metricsUrl = url;
    return originalXHROpen.apply(this, [method, url, ...rest]);
  };

  XMLHttpRequest.prototype.send = function(...args) {
    if (isAnalyticsUrl(this._metricsUrl)) {
      this.addEventListener('load', function() {
        try {
          const data = JSON.parse(this.responseText);
          processApiResponse(this._metricsUrl, data);
        } catch (e) {
          // Ignore parsing errors
        }
      });
    }
    return originalXHRSend.apply(this, args);
  };

  /**
   * Check if URL is analytics-related
   */
  function isAnalyticsUrl(url) {
    if (!url) return false;
    const patterns = [
      'TweetActivityQuery',
      'TweetDetail',
      '/activity/',
      'UserTweets'
    ];
    return patterns.some(p => url.includes(p));
  }

  /**
   * Process API response and extract metrics
   */
  function processApiResponse(url, data) {
    console.log('[TwitterMetrics] Processing API response:', url);

    // Handle TweetActivityQuery responses (analytics popup data)
    if (url.includes('TweetActivityQuery') && data?.data?.tweet_activity_v2) {
      const activity = data.data.tweet_activity_v2;
      const tweetId = activity.tweet_id || extractTweetIdFromUrl();

      if (tweetId) {
        const metrics = {
          tweet_id: tweetId,
          impressions: activity.impression_count,
          engagements: activity.engagement_count,
          detail_expands: activity.detail_expand_count,
          profile_visits: activity.user_profile_click_count,
          link_clicks: activity.url_click_count,
          // These might also be in the response
          likes: activity.favorite_count,
          retweets: activity.retweet_count,
          replies: activity.reply_count,
          quotes: activity.quote_count,
          bookmarks: activity.bookmark_count
        };

        sendMetrics(metrics);
      }
    }

    // Handle TweetDetail responses (tweet page data)
    if (url.includes('TweetDetail') && data?.data?.tweetResult?.result) {
      const tweet = extractTweetFromResult(data.data.tweetResult.result);
      if (tweet) {
        sendMetrics(tweet);
      }
    }

    // Handle nested tweet data structures
    if (data?.data?.user?.result?.timeline_v2?.timeline?.instructions) {
      const instructions = data.data.user.result.timeline_v2.timeline.instructions;
      for (const instruction of instructions) {
        if (instruction.entries) {
          for (const entry of instruction.entries) {
            const tweet = findTweetInEntry(entry);
            if (tweet) {
              sendMetrics(tweet);
            }
          }
        }
      }
    }
  }

  /**
   * Extract tweet data from API result object
   */
  function extractTweetFromResult(result) {
    if (!result) return null;

    // Handle TweetWithVisibilityResults wrapper
    if (result.__typename === 'TweetWithVisibilityResults') {
      result = result.tweet;
    }

    if (result.__typename !== 'Tweet') return null;

    const legacy = result.legacy;
    if (!legacy) return null;

    const metrics = {
      tweet_id: legacy.id_str || result.rest_id,
      text: legacy.full_text,
      author_username: result.core?.user_results?.result?.legacy?.screen_name,
      created_at: legacy.created_at ? new Date(legacy.created_at).toISOString() : null,
      likes: legacy.favorite_count || 0,
      retweets: legacy.retweet_count || 0,
      replies: legacy.reply_count || 0,
      quotes: legacy.quote_count || 0,
      bookmarks: legacy.bookmark_count || 0
    };

    return metrics;
  }

  /**
   * Find tweet data in timeline entry
   */
  function findTweetInEntry(entry) {
    if (!entry?.content?.itemContent?.tweet_results?.result) {
      return null;
    }
    return extractTweetFromResult(entry.content.itemContent.tweet_results.result);
  }

  /**
   * Extract tweet ID from current URL
   */
  function extractTweetIdFromUrl() {
    const match = window.location.pathname.match(/\/status\/(\d+)/);
    return match ? match[1] : null;
  }

  /**
   * Send metrics to background script
   */
  function sendMetrics(metrics) {
    // Filter out undefined/null values
    const cleanMetrics = {};
    for (const [key, value] of Object.entries(metrics)) {
      if (value !== undefined && value !== null) {
        cleanMetrics[key] = value;
      }
    }

    if (!cleanMetrics.tweet_id) {
      console.log('[TwitterMetrics] No tweet_id, skipping');
      return;
    }

    console.log('[TwitterMetrics] Sending metrics:', cleanMetrics);
    chrome.runtime.sendMessage({
      type: 'METRICS_DATA',
      data: cleanMetrics
    });
  }

  /**
   * Initialize: detect username after page loads
   */
  function init() {
    // Try to detect username immediately
    let username = detectUsername();

    // If not found, try again after a delay (page might still be loading)
    if (!username) {
      setTimeout(() => {
        username = detectUsername();
        if (!username) {
          // Set up observer for dynamic content
          const observer = new MutationObserver(() => {
            username = detectUsername();
            if (username) {
              observer.disconnect();
            }
          });
          observer.observe(document.body, { childList: true, subtree: true });

          // Stop observing after 30 seconds
          setTimeout(() => observer.disconnect(), 30000);
        }
      }, 2000);
    }
  }

  // Run init when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
