/**
 * Background service worker for Twitter Metrics Tracker
 * Intercepts X.com API responses to capture tweet analytics
 */

// Configuration - update this with your backend URL and API key
const CONFIG = {
  backendUrl: 'https://twitter-metrics.example.com',  // TODO: Update with actual URL
  apiKey: 'tmk_change_this_to_a_secure_random_key'    // TODO: Update with actual key
};

// Store for pending metrics to batch/dedupe
const pendingMetrics = new Map();
let sendTimeout = null;

/**
 * Listen for completed requests to X's API
 */
chrome.webRequest.onCompleted.addListener(
  async (details) => {
    // Only process successful JSON responses
    if (details.statusCode !== 200) return;

    // Check if this is an analytics-related endpoint
    const url = details.url;
    if (!isAnalyticsEndpoint(url)) return;

    // We can't read response body directly from webRequest in Manifest V3
    // The content script will need to intercept fetch/XHR responses
    console.log('[TwitterMetrics] Detected analytics request:', url);
  },
  {
    urls: [
      'https://api.x.com/*',
      'https://api.twitter.com/*',
      'https://x.com/i/api/*',
      'https://twitter.com/i/api/*'
    ]
  }
);

/**
 * Check if URL is an analytics-related endpoint
 */
function isAnalyticsEndpoint(url) {
  const analyticsPatterns = [
    'TweetActivityQuery',
    'TweetDetail',
    'UserTweets',
    'activity/about_me'
  ];
  return analyticsPatterns.some(pattern => url.includes(pattern));
}

/**
 * Handle messages from content script
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'METRICS_DATA') {
    handleMetricsData(message.data);
    sendResponse({ status: 'received' });
  } else if (message.type === 'GET_CONFIG') {
    sendResponse({ config: CONFIG });
  } else if (message.type === 'UPDATE_CONFIG') {
    Object.assign(CONFIG, message.config);
    chrome.storage.local.set({ config: CONFIG });
    sendResponse({ status: 'updated' });
  } else if (message.type === 'GET_USERNAME') {
    chrome.storage.local.get(['username'], (result) => {
      sendResponse({ username: result.username });
    });
    return true; // Keep channel open for async response
  } else if (message.type === 'SET_USERNAME') {
    chrome.storage.local.set({ username: message.username });
    console.log('[TwitterMetrics] Username set:', message.username);
    sendResponse({ status: 'saved' });
  }
  return true;
});

/**
 * Handle metrics data from content script
 */
function handleMetricsData(data) {
  if (!data.tweet_id) return;

  // Get current username filter
  chrome.storage.local.get(['username'], async (result) => {
    const myUsername = result.username;

    // If we have a username filter, only track our own tweets
    if (myUsername && data.author_username) {
      if (data.author_username.toLowerCase() !== myUsername.toLowerCase()) {
        console.log('[TwitterMetrics] Skipping tweet from other user:', data.author_username);
        return;
      }
    }

    // Dedupe: use tweet_id as key, keep most recent data
    pendingMetrics.set(data.tweet_id, {
      ...data,
      captured_at: Date.now()
    });

    // Debounce sending to backend
    if (sendTimeout) clearTimeout(sendTimeout);
    sendTimeout = setTimeout(sendPendingMetrics, 2000);

    console.log('[TwitterMetrics] Queued metrics for tweet:', data.tweet_id);
  });
}

/**
 * Send pending metrics to backend
 */
async function sendPendingMetrics() {
  if (pendingMetrics.size === 0) return;

  const metrics = Array.from(pendingMetrics.values());
  pendingMetrics.clear();

  for (const data of metrics) {
    try {
      const response = await fetch(`${CONFIG.backendUrl}/api/metrics`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-API-Key': CONFIG.apiKey
        },
        body: JSON.stringify(data)
      });

      if (response.ok) {
        console.log('[TwitterMetrics] Sent metrics for tweet:', data.tweet_id);
      } else {
        console.error('[TwitterMetrics] Failed to send metrics:', response.status);
      }
    } catch (error) {
      console.error('[TwitterMetrics] Error sending metrics:', error);
    }
  }
}

/**
 * Load saved config on startup
 */
chrome.storage.local.get(['config'], (result) => {
  if (result.config) {
    Object.assign(CONFIG, result.config);
    console.log('[TwitterMetrics] Loaded config from storage');
  }
});

console.log('[TwitterMetrics] Background service worker started');
